function fdcinit()
% FDCINIT - toolbox initialization utility
%
% This program temporarily enhances the Matlab search-path with the FDC 
% root- and subdirectories. Usually, this routine is started via FDC.M,
% which is present in the FDC root-directory. To facilitate FDC initia-
% lization for future sessions, it is recommended to include the FDC
% root-directory to the Matlab path permanently.
%
% FDCINIT allows the user to change the names of the FDC root-directory,
% and to add, rename or delete FDC subdirectories. This option is useful
% if the toolbox is used as a starting point for the implementation of 
% other models and tools. 
%
% The standard tree-structure for the FDC-package is:
%
% Default root-directory:
%           MATLABROOT\TOOLBOX\FDC14\
%
% (note: FDC 1.4 applies generic path and file-separators, even
% though the documentation below assumes MS-DOS path-notation).
%
% Default subdirectories:
%           AIRCRAFT  contains the general aircraft model
%           APILOT    contains the autopilot simulation models
%           DATA      used for storing model parameters and other
%                     datafiles (e.g. trim conditions)
%           DOC       contains program documentation (README-files)
%           EXAMPLES  contains open-loop FDC examples
%           HELP      contains help texts for the FDC models
%           NAVIGATE  contains a library with VOR/ILS blocks
%           PROGRAMS  contains all Matlab programs from the FDC
%                     toolbox, except the ones which are clearly
%                     related to one of the other subdirectories
%           TOOLS     contains the FDCTOOLS library
%           WIND      contains a wind and turbulence library
%
% When FDCINIT is used for the first time, this standard path will be 
% used as starting point for the Matlab-path extension for FDC. The path 
% may be changed if required, using the menu options from FDCINIT. Any 
% user-specified changes to this path will be saved to the file FDC.INI, 
% and will be used as new default path-setting the next time FDCINIT is 
% started. The FDC.INI file is saved to the same directory as fdcinit.m 
% (i.e. the PROGRAMS subdirectory).
%
% Normally, when FDCINIT is started, the FDC tree will be displayed, and 
% the user will be asked whether the path needs to be changed each time 
% FDCINIT is run. If desired, that message can be suppressed for future 
% sessions by pressing 's' for 'suppress' in the opening menu. If that 
% option is used inadvertently, the default path can be restored by 
% deleting the file FDC.INI from the PROGRAMS subdirectory.
%
% Note: FDC.INI is also used by the FDCDIR utility, which in turn is used
% by several other utilities from the FDC toolbox, to define a default 
% directory for e.g. datafiles and helpfiles. If FDCDIR cannot find the 
% FDC.INI file, it will use the DEFAULT path definition, given above.

clc
errorflag = 0;

if exist('fdc.ini') == 0   % Path definition has not been created yet,
                           % use default definitions as starting point
                           % -----------------------------------------
                           
   % The root directory of the Matlabpath extension (i.e., the FDC root) is
   % stored in rootdir. MATLABROOT\TOOLBOX\FDC14 is the default definition.
   % ----------------------------------------------------------------------
   rootdir = fullfile(matlabroot,'toolbox','fdc14');

   % The definition of the subdirectories of the FDC package are stored in the cell-
   % array subdirs (directory = directory). Default definitions are given below:
   % ---------------------------------------------------------------------------
   subdirs = {'aircraft','apilot','data','doc','examples','help', ...
              'navigate','programs','tools','wind'};

   % The default definition of the Matlabpath extension will be used and the
   % user will be asked if that's correct (if not, the Matlab-path extension
   % can be changed). This question will appear every next time FDCINIT is
   % used, but it is possible to suppress the message for future sessions.
   % The variable 'suppress' is used to check this. If suppress = 0 (default)
   % the "is this correct" message will appear when starting FDCINIT; if the
   % user selects the option "suppress this message", suppress = 1 will be
   % set, and the message will not appear the next times FDCINIT is started.
   % Redefining the directory tree afterwards is then only possible after
   % deleting the file FDC.INI before running FDCINIT again.
   % ------------------------------------------------------------------------
   suppress = 0;

else

   % Load definition of the Matlabpath extension and suppress 
   % variable from the initialization file FDC.INI.
   % --------------------------------------------------------
   load fdc.ini -mat

end

% Display path information, unless user has selected 'suppress' option
% during an earlier FDC session.
% --------------------------------------------------------------------
if suppress ~= 1

   clc
   
   disp(' ');
   disp('The Flight Dynamics and Control toolbox');
   disp('=======================================');
   disp(' ');

   if exist('fdc.ini') == 0  % First time use: show all directory details

      disp('Please verify the FDC path-settings. Current FDC path:');
      disp(' ');
      disp('FDC root-directory:');
      disp('-------------------');
      disp(['   ',matlabroot,filesep,'TOOLBOX',filesep,'FDC14']);
      disp(' ');
      disp('FDC subdirectories:');
      disp('-------------------');
      disp('   AIRCRAFT    nonlinear aircraft model and model libraries');
      disp('   APILOT      autopilot simulation models');
      disp('   DATA        model parameters and other FDC datafiles');
      disp('   EXAMPLES    examples and tutorials');
      disp('   HELP        on-line help texts');
      disp('   NAVIGATE    radio-navigation library (VOR, ILS)');
      disp('   PROGRAMS    Matlab macros and functions for the FDC toolbox');
      disp('   TOOLS       generic blocklibrary');
      disp('   WIND        wind and turbulence blocks');
      disp(' ');
      disp('By default this definition should be allright. Change only if');
      disp('really neccessary!');
      disp(' ');

   else % FDCINIT already has been used at least one time; show current 
        % directory list (FDC path) without additional comments

      disp('Current FDC path:');
      disp(' ');
      disp('FDC root directory:');
      disp('-------------------');
      disp(rootdir);
      disp(' ');
      disp('FDC subdirectories:');
      disp('-------------------');
      disp(char(subdirs));
      disp(' ');

   end

   % Main path-verification loop
   % ---------------------------
   ready = 0;
   while ready~=1

      % Verify current FDC path
      % -----------------------
      answ=input('FDC path ok? ([y]=yes, n=no, s=suppress this message in future): ','s');

      if isempty(answ)
         answ = 'y';
      end

      if answ == 'n'
      
         % Change Matlabpath extension. The 'while ready1~=1' loop is used to 
         % continue returning to the path-edit Options menu, until all required
         % changes have been made.
         % --------------------------------------------------------------------
         ready1 = 0;
         while ready1 ~= 1
            clc
            opt = txtmenu('FDC path edit-options','Change FDC root directory',...
                          'Rename or delete FDC subdirectories',...
                          'Add FDC subdirectories','Ready');

            if opt == 1        % Change root directory
                               % ---------------------
               disp(' ');
               disp(' ');
               disp(['Current FDC root: ' rootdir]);
               disp(' ');

               % Enter new name of root directory (including root drive)
               % ----------------------------------------------------
               newroot = input('New root directory: ','s');
               if ~isempty(newroot)
                  rootdir = newroot;
                  clear newroot
               end

            elseif opt == 2    % Rename or delete subdirectories
                               % -------------------------------

               m = length(subdirs);

               disp(' ');
               disp(' ');

               % newsubs = temporary cell-array to store changed subdirectory tree
               % -----------------------------------------------------------------
               newsubs = {};

               % i = index into subdirs cell-array
               % j = index into newsubs cell-array
               % ---------------------------------
               j = 0;

               for i = 1:m     % For all subdirectories: ask if changes have to
                               % be made, if the subdir has to be deleted from
                               % the path, or if old subdirectory must be used.
                               % ----------------------------------------------
                  disp(' ');
                  disp(' ');
                  disp(['Subdirectory ' num2str(i) ': ' subdirs{i}]);
                  disp(' ');

                  answ2 = input('[s] = skip, r = rename, d = delete from path: ','s');
                  if isempty(answ2)
                     answ2 = 's';
                  end

                  if answ2 == 'r'
                     newname = input('New subdirectory name: ','s');
                     if isempty(newname)
                        newname = subdirs{i};
                     end

                     j = j+1;

                     % New subdirectory stored into cell-array newsubs
                     % -----------------------------------------------
                     newsubs{j} = newname;

                  elseif answ2 == 'd'

                     % Don't store a directoryname in newsubs (has the same
                     % effect as deleting a directoryname from the variable
                     % subdirs!).
                     % -------------------------------------------------
                     disp(' ');
                     disp(['subdirectory ' subdirs{i} ' deleted!']);

                  else
                     j = j+1;

                     % Store old subdirectory from FDC.INI or the default 
                     % directories (used if FDC.INI does not yet exist)
                     % without changes into the cell-array newsubs.
                     % --------------------------------------------------
                     newsubs{j} = subdirs{i};

                  end

               end

               % Replace old directory tree by the new tree stored in newsubs
               % ------------------------------------------------------------
               subdirs = newsubs;
               m = length(subdirs); % To prevent errors when subdirectory-
                                    % change code is called again directly
                                    % after finishing THIS subdir-change.
               disp(' ');
               disp('Current definition of subdirectory tree:');
               disp(' ');
               disp(char(subdirs));
               disp(' ')
               disp(' ');
               disp('<<< Press a key >>>');
               pause

            elseif opt == 3    % Add subdirectories to tree

               ready2 = 0;
               disp(' ');
               while ready2 ~= 1
                  clc
                  disp('Current definition of subdirectory tree:');
                  disp(' ');
                  disp(char(subdirs));
                  disp(' ');
                  disp(' ');
                  disp('Enter name of subdirectory to add (press Enter to return to menu):');
                  disp(' ');

                  % Enter new subdirectory name.
                  % -------------------------
                  newname = input('> ','s');

                  % If new entry is empty, no new directories will be added
                  % to the directory tree anymore.
                  % -------------------------------------------------------
                  if isempty(newname)
                     ready2 = 1;
                  end

                  % Add new directory entry to subdirectory-tree
                  % --------------------------------------------
                  if ready2 ~= 1
                     subdirs{m+1} = newname;
                  end

                  m = length(subdirs); % To prevent errors when subdirectory-
                                       % enhancement code is called again
                                       % directly after finishing THIS
                                       % subdir-enhancement.
               end

            elseif opt == 4    % Ready, current Matlabpath extension should be ok, however
                               % final check will be done by displaying the current path
                               % here and jumping back to the while ready~=1 line.
                               % ---------------------------------------------------------
               clc
               disp('Current FDC path:');
               disp(' ');
               disp('FDC root directory:');
               disp('-------------------');
               disp(rootdir);
               disp(' ');
               disp('FDC subdirectories:');
               disp('-------------------');
               disp(char(subdirs));
               disp(' ');

               ready1 = 1;

            end

         end  % of path-edit options menu loop 

         % Save results to FDC.INI file
         % --------------------------------
         save fdc.ini rootdir subdirs suppress

      elseif answ == 's'

         % The displayed path should now be correct, but checking anyway
         % -------------------------------------------------------------
         if exist(rootdir) ~= 7
            errormsg = ['The specified root-directory does not exist! ', ...
                        'Please check your FDC installation, and re-run FDC.M with ', ...
                        'the correct root-directory.'];
            h = newMsgBox(errormsg,'ERROR');
            uiwait(h);
            clear h errormsg;
            errorflag = 1;

         else

            % Don't change Matlabpath extension, and don't display the "Is this
            % correct?" message anymore in the future.
            % -----------------------------------------------------------------
            disp(' ');
            disp('If you select this option, the current setting for the Matlab-');
            disp('path-extension for the FDC toolbox will be used for _all_');
            disp('future FDC sessions. The path-definition can _only_ be changed');
            disp('with FDCINIT again after deleting the file FDC.INI from');
            disp('the FDC root-directory!');
            disp(' ');
            disp('Are you sure you want to suppress the FDC path-verification');
            disp('for future sessions ([y]/n)?');
            disp(' ');
            answ1 = input('?> ','s');
            if isempty(answ1)
               answ1 = 'y';
            end
            if answ1 == 'y'
               disp(' ');
               disp('So be it...');
               disp(' ');
               disp('<<< Press a key to continue>>>');
               pause

               % Change suppress to 1, to make sure that the directory tree in
               % FDC.INI will be used as Matlabpath extension from now on
               % (every time FDCINIT is started).
               % -------------------------------------------------------------
               suppress = 1;

               % Save results to FDC.INI file
               % ----------------------------
               save fdc.ini rootdir subdirs suppress
            end

            ready = 1;

         end

      else  
         % The displayed path should now be correct, but checking anyway
         % -------------------------------------------------------------
         if exist(rootdir) ~= 7
            errormsg = ['The specified root-directory does not exist. ', ...
                        'Please check your FDC installation, and re-run FDC.M with ', ...
                        'the correct root-directory.'];
            h = newMsgBox(errormsg,'ERROR');
            uiwait(h);
            clear h errormsg;
            errorflag = 1;
         end

         if exist('fdc.ini') == 0

            % Save results to FDC.INI file. This is not necessary if FDC.INI 
            % already exists, since the current setting will be used.
            % --------------------------------------------------------------
            save fdc.ini rootdir subdirs suppress

         end
         
         ready = 1;
         
      end
   end % main path-verification loop
end 


% Add new directories to Matlabpath. Include the FDC root-directory itself 
% only if it doesn't exist already. (If the user follows the installation
% guidelines, the FDC root-directory will be added to the default definition
% of the Matlabpath, while the FDC subdirectories are added by FDCINIT.
% Duplicate names in the Matlabpath will create warning messages, which
% are thus quite likely for the FDC root-directory but not for the sub-
% directories. For this reason, checking FDC subdirectories for duplicate
% names was NOT considered necessary.)
% --------------------------------------------------------------------------
OS = computer;
if strcmp(OS,'PCWIN')
   pos = findstr(lower(matlabpath),lower(rootdir));
else
   pos = findstr(matlabpath,rootdir);
end

m = length(subdirs); % Update, because subdirs may have been changed.
if isempty(pos)
   path(path,rootdir);
end
for i = 1:m
   path(path,fullfile(rootdir,deblank(char(subdirs(i)))));
end

if errorflag ~= 0
    disp(' ');
    delete('fdc.ini');
    error('There were errors. Please verify your FDC root-directory and run FDC again!');
end

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004. 
%-----------------------------------------------------------------------